define([
    'underscore',
    'backbone',
    'models/abstract-model',
], function(_, Backbone, Model) {
    'use strict';

    describe('Abstract Model', function() {
        describe('Native Function', function() {
            it('binds the url function', function() {
                var M = Model.extend({
                    url: function() {
                        return this.get('works');
                    },
                });
                var model = new M({works: 'test'});
                var url = _.result(model, 'url');

                expect(url).toBe('test');
            });

            it('listens for sync', function() {
                var model = new Model();
                var isSyncSet = false;
                model.listenTo(model, 'fetch:success', function() {
                    isSyncSet = true;
                });
                model.trigger('sync');

                expect(isSyncSet).toBeTruthy();
            });

            it('listens for error', function() {
                var model = new Model();
                var isErrorSet = false;
                model.listenTo(model, 'fetch:error', function() {
                    isErrorSet = true;
                });
                model.trigger('error');

                expect(isErrorSet).toBeTruthy();
            });

            it('will return an existing resource', function() {
                var model = new Model();
                var existingResource = model.getResource('appointment-requests');
                var nonExisting = model.getResource('this-does-not-exist-and-never-will');

                expect(_.isFunction(existingResource.formatHref)).toBeTruthy();
                expect(nonExisting).toBeUndefined();
            });

            describe('fetch calls', function() {
                var calledWith;

                beforeEach(function() {
                    spyOn(Backbone.Model.prototype, 'sync').and.callFake(function() {
                        calledWith = Array.prototype.slice.call(arguments);
                        this.trigger.apply(this, arguments);
                    });
                });

                it('adds default options to a fetch request', function() {
                    var model = new Model();
                    var options;

                    model.fetch();
                    options = calledWith[2];

                    expect(options.contentType).toBe('application/json');
                    expect(options.dataType).toBe('json');
                    expect(options.traditional).toBe(true);
                });

                it('adds default options to a save request', function() {
                    var model = new Model();
                    var options;

                    model.save();
                    options = calledWith[2];

                    expect(options.contentType).toBe('application/json');
                    expect(options.dataType).toBe('json');
                    expect(options.traditional).toBe(true);
                });

                it('honors the supplied version over the defaults on a fetch', function() {
                    var model = new Model();
                    var options;

                    model.fetch({dataType: 'xml'});
                    options = calledWith[2];

                    expect(options.contentType).toBe('application/json');
                    expect(options.dataType).toBe('xml');
                    expect(options.traditional).toBe(true);
                });
            });
        });
    });
});
